using System;
using System.Xml;
using System.Collections.Specialized;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class is a collection of ServerCollectionInfo objects identified and accessed
	/// by server handle. It is used to access list of servers with theirs connection information
	/// stored in application configuration file.
	/// </summary>
	public class ServerConnectionInfoCollection : NameObjectCollectionBase
	{
		/// <summary>
		/// Name of root element in XML fragment representing 
		/// deserialized <see cref="ServerConnectionInfoCollection"/>.
		/// </summary>
		internal const string CollectionXmlRootElementName = "ServerList";

		private ServerConnectionInfoCollection() : 
			base() {}

		/// <summary>
		/// XML deserialization factory method parsing supplied XML node 
		/// and returning an instance of ServerConnectionInfoCollection created from it.
		/// </summary>
		/// <param name="serverInfoCollectionNode">
		///		XML node containing containing server connection info collection data.
		///	</param>
		/// <returns>New instance of ServerConnectionInfoCollection object deserialized from a given node.</returns>
		public static ServerConnectionInfoCollection Parse( XmlNode serverInfoCollectionNode )
		{
			if( serverInfoCollectionNode == null )
				throw( new ArgumentNullException( "serverInfoCollectionNode" ) );

			XmlUtility.ParseValidateConvertNodeToElement( serverInfoCollectionNode, CollectionXmlRootElementName );
		
			ServerConnectionInfoCollection _result = new ServerConnectionInfoCollection();

			foreach( XmlNode _serverNode in serverInfoCollectionNode.ChildNodes )
			{
				if( _serverNode.NodeType != XmlNodeType.Element )
					continue; 

				try
				{
					_result.Add( ServerConnectionInfo.Parse( _serverNode ) );
				}
				catch( ArgumentException xcp ) // there may be number of reasons to fail
				{
					throw( new XmlParseException( SR.Exceptions.XmlDeserializationFailedDueToInnerException( 
						typeof(ServerConnectionInfoCollection).Name, _serverNode.Name ), xcp ) );				
				}			
			}

			return _result;
		}

		/// <summary>
		/// Retrieves server connection info by server's handle.
		/// </summary>
		/// <param name="handle">Internal handle indentifying server within the client application.</param>
		/// <returns>An instance of ServerConnectionInfo mapped to a given handle. Null if nothing is found.</returns>
		public ServerConnectionInfo GetServerConnectionInfoByHandle( string handle )
		{
			if( handle == null )
				throw( new ArgumentNullException( "handle" ) );

			return (ServerConnectionInfo)BaseGet( handle );
		}

		/// <summary>
		/// Adds a given instance of ServerConnectionInfo to a collection.
		/// </summary>
		/// <param name="serverConnectionInfo">Server connection info to add to collection.</param>
		/// <returns>Reference to added object.</returns>
		private ServerConnectionInfo Add( ServerConnectionInfo serverConnectionInfo )
		{
			if( serverConnectionInfo == null )
				throw( new ArgumentNullException( "serverConnectionInfo" ) );

			if( serverConnectionInfo.Handle == null )
				throw( new ArgumentOutOfRangeException( 
					"serverConnectionInfo", 
					SR.Exceptions.ServerConnectionInfoCollectionNullHandle( typeof( ServerConnectionInfo ).GetType().Name ) ) );

			if( BaseGet( serverConnectionInfo.Handle ) != null )
				throw( new ArgumentException( SR.Exceptions.ServerConnectionInfoCollectionDuplicateHandle( 
					typeof( ServerConnectionInfo ).GetType().Name, serverConnectionInfo.Handle ), "serverConnectionInfo" ) );
				
			BaseAdd( serverConnectionInfo.Handle, serverConnectionInfo );

			return serverConnectionInfo;
		}
	}
}
